function [p,x] = PPWdensity(model,x,x0,dt,theta,method)
%PPWdensity : This code calculates the epsilon-expansion approximation of
%the transition density of a 1d SDE as described in the accompanying paper:
%Approximate MLE for 1D diffusions on a fine grid (Lu, Paine, Preston, Wood, 2020)
%
% Input:
% model : SDE of interest. One of cir, ou, blackscholes, bessel, ginzland
%These are just ready-to-use examples.
% x : vector of points to calculate density at
% x0 : initial data point
% dt : interval between x0 and x
% theta : vector of parameters 
% method : Transition denisty approximation followed by _u or _t. First, the
% method is one of euler, elerian, epsilon1, epsilon1b and epsilon2 and
% this is followed by _u or _t indicating if the model has been
% transformed to unit diffusion (_t) or not (_u). Only euler and elerian
% accept _u as the suffix.
%
% Output : 
% p : vector of density values for the specified inputs.
% x : vector of points evaluated at
%
% Example:
% Calculate the epsilon expansion transition density approximation with 
% O(T^2) correction terms for the Ornstein-Uhlenbeck process with 
% parameters (0.1,0.2,0.02), initial point x0 = 0.1, future points equally 
% spaced between 0.05 and 0.15 and interval between x0 and x of 1/12.
% dt = 1/12;
% x = linspace(0.05,0.15,10);
% x0 = 0.1;
% theta = [0.1;0.2;0.02];
% [p,x] = PPWdensity('OU',x,x0,dt,theta,'epsilon2_t');
%% Calculating the derivatives of the drift function (and diffusion if _u used)

switch model
    case 'ou'
        % Gamma transform
        kappa = theta(1); alpha = theta(2); beta = theta(3);
        y = x/beta;
        y0 = x0/beta;
        sigmaOfx = beta*(ones(size(x)));
        
        % Deal coeffs
        a0 = kappa*alpha/beta-kappa*y0;
        a1 = -kappa;
        a2 = 0;
        a3 = 0;
        a4 = 0;
        b0 = 1;
    case 'cir'
        % Gamma transform
        kappa = theta(1); alpha = theta(2); beta = theta(3);
        y = 2*sqrt(x)/beta;
        y0 = 2*sqrt(x0)/beta;
        sigmaOfx = beta*sqrt(x);
        
        % Deal coeffs
        foo = 2*kappa*alpha/beta^2-1/2;
        a0 = foo/y0 - kappa/2*y0;
        a1 = -foo/y0^2 - kappa/2;
        a2 = 2*foo/y0^3;
        a3 = -6*foo/y0^4;
        a4 = 24*foo/y0^5;
        b0 = 1;
end
  
%% Calculation of the density approximation specified by method
        R= y0+a0*dt+(1/2)*(a0*a1 + a2* 1/2)*dt^2 + (1/6)*(a0*(a0*a2 ...
            +(a1)^2 +a3*1/2)+ (1/2)*(a0*a3+3*a1*a2)+a4*1/4)*dt^3;
        z= (y-R)/sqrt(dt) ;
        phi = 1/(sqrt(2*pi)) * exp(-(z.^2)/2) ;
        C1110 = a3;
        C100 = (a0*a2 + a1^2 + 1/2*a3);
        C010 = a0*( a2) + 1/2*( a3);
        H2 = z.^2 - 1 ;
        H3 = z.^3 - 3.*z;
        H4 = z.^4 - 6.*z.^2 + 3;
        A = 1/2.*a1.*H2;
        B = 1/6.*a2.*H3  ;
        C = (a1)^2.*(1/8.*H4 + 1/6.*H2) + 1/6*(C100 + C010).*H2 + ...
            1/24*C1110*H4; 
        py = phi .* ( 1 + (dt).*A + (dt)^(3/2).*B + (dt)^2.*C).*1/sqrt(dt);      


p=py./sigmaOfx; %Transform from unit diffusion to original SDE if necessary

end