function [p,x] = densityExact(model,x,x0,dt,theta)
%densityExact : calculates the transition density of SDEs where it is known
% exactly.
%
% Input: 
%
% model : one of ou, cir, toy and bessel
% x : points at which to evaulate the transition density
% x0 : initial value 
% dt : interval over which to evaluate transition density
% theta : parameters that specify the model. We use the parameterisations:
% OU : dXt = theta(1) ( theta(2) - Xt) dt + theta(3) dWt
% CIR : dXt = theta(1) ( theta(2) - Xt) dt + theta(3) sqrt(Xt) dWt
% 
%
% Output:
%
% p : transition density value
%
% x : points at which we evaluated the density (same as input)
%
% Example:
% Calculate the transition density for the Ornstein-Uhlenbeck process with 
% parameters (0.1,0.2,0.02), initial point x0 = 0.1, future points equally 
% spaced between 0.05 and 0.15 and interval between x0 and x of 1/12.
% dt = 1/12;
% x = linspace(0.05,0.15,10);
% x0 = 0.1;
% theta = [0.1;0.2;0.02];
% [p,x] = densityExact('OU',x,x0,dt,theta);


switch lower(model)
   case 'ou'
      if nargin < 2 % by default, plot density corresponding to AS (1999, p.1370)
         x = linspace(0.08,.12,100); x0 = 0.1; dt = 1/12;
         theta = [.0717 .261 .02237];
      end
      kappa = theta(1);
      alpha = theta(2);
      beta = theta(3);
      gammaSq = beta^2*(1-exp(-2*kappa*dt));
      p = 1/sqrt(pi*gammaSq/kappa)*exp(-(x-alpha-(x0-alpha)*...
         exp(-kappa*dt)).^2*kappa/gammaSq);
   case 'cir'
      if nargin < 2 % by default, plot density corresponding to AS (1999, p.1373)
         x = linspace(.043,.077,100); x0 = .06; dt = 1/12;
         theta = [.0721 .219 .06665];
      end
      % The following uses the expression given in AS (1999, p.1371) and
      % the function NCX2PDF in Matlab's statistics toolbox to evaluate
      % the non-central chi-square density
      kappa = theta(1);
      alpha = theta(2);
      beta = theta(3);
      q = 2*kappa*alpha/beta^2 - 1;
      c = 2*kappa/(beta^2*(1-exp(-kappa*dt)));
      u = c*x0*exp(-kappa*dt);
      v = c*x;
      p = (2*c)*ncx2pdf(2*v,2*(q+1),2*u);
   case 'toy'
      p = normpdf(x,x0+theta*dt,sqrt(dt));
   case 'bessel'
      % see Revuz and Yor (1999) p.441
      kappa = theta;  % =delta in R&Y, 1999
      nu = kappa/2 - 1;
      p = 1/(2*dt)*(x/x0).^(nu/2).*exp(-(x+x0)/(2*dt)).*...
         besseli(nu,sqrt(x*x0)/dt);
   otherwise
      error(...
         '''model'' must be one of the prescribed models (e.g. ''cir''');
end



