function [] = plotPointsOnSphere(Y,marker_colour,marker_size)
%plotPointsOnSphere : plots data points that lie on the unit sphere S^2 
% as small discs on the surface of the sphere.  Looks much nicer and is
% less temperamental than just using plot3.
%
%INPUTS:
% Y : 3 x n matrix of data, each column of which is a unit vector,
%     where n is the sample size.
%
% marker_colour : colour of the markers, using standard Matlab syntax 
%  for specifying colours [default = 'r', i.e. red]
%
% marker_size : size of markers, in arbitrary units [default = 1]


if nargin < 3, marker_size = 1; end
if nargin < 2, marker_colour = 'r'; end

phi360 = linspace(-pi,pi,50);
scale_of_radius = 1.02; % put points marginally above spherical surface

figure(gcf); hold on;
for i1 = 1:size(Y,2),
  n = Y(:,i1)/norm(Y(:,i1));
  Y_perp = cross([1,0,0]',Y(:,i1));
  Ytmp = Y(:,i1);
  Ytmp = rot_around_v(Y_perp/norm(Y_perp),marker_size*pi/180)*Y(:,i1);
  for i2 = 1:length(phi360),
    Ypatch(:,i2) = rot_around_v(n,phi360(i2))*(Ytmp*scale_of_radius);
  end
  fill3(Ypatch(1,:),Ypatch(2,:),Ypatch(3,:),marker_colour,'LineWidth',0.3)
end


function R = rot_around_v(e_rot,phi)
% ROT_AROUND_V - matrix for rotation PHI radians around E_ROT
%
% Calling:
% R = rot_around_v(e_rot,phi)

% Copyright � 2002 Bjorn Gustavsson, <bjorn.gustavsson@irf.se>
% This is free software, licensed under GNU GPL version 2 or later

r = [cos(phi) -sin(phi) 0;
     sin(phi) cos(phi) 0;
     0 0 1];
e_Rot = e_rot./sum(e_rot.^2);
az = atan2(e_Rot(1),e_Rot(2));
ze = acos(e_Rot(3));
T = [1 0 0;
     0 cos(ze) -sin(ze);
     0 sin(ze) cos(ze)]*...
    [cos(az) -sin(az) 0;
     sin(az) cos(az) 0;
     0 0 1];

R = T'*r*T;