function [p,x] = densityExact(model,x,x0,dt,theta)

% DENSITYEXACT  The exact transition density for various SDEs 
%    (Ornstein-Uhlenbeck, Cox-Ingersol-Ross, Bessel).
%
% Example:
%    f = densityExact('CIR',linspace(.05,.15,100),0.1,1/12,[.5 .06 .15]);

% Written by Simon Preston (http://www.maths.nott.ac.uk/~spp), 2009

switch lower(model)
   case 'ou'
      if nargin < 2 % by default, plot density corresponding to AS (1999, p.1370)
         x = linspace(0.08,.12,100); x0 = 0.1; dt = 1/12;
         theta = [.0717 .261 .02237];
      end
      kappa = theta(1);
      alpha = theta(2);
      beta = theta(3);
      gammaSq = beta^2*(1-exp(-2*kappa*dt));
      p = 1/sqrt(pi*gammaSq/kappa)*exp(-(x-alpha-(x0-alpha)*...
         exp(-kappa*dt)).^2*kappa/gammaSq);
   case 'cir'
      if nargin < 2 % by default, plot density corresponding to AS (1999, p.1373)
         x = linspace(.043,.077,100); x0 = .06; dt = 1/12;
         theta = [.0721 .219 .06665];
      end
      % The following uses the expression given in AS (1999, p.1371) and
      % the function NCX2PDF in Matlab's statistics toolbox to evaluate
      % the non-central chi-square density
      kappa = theta(1);
      alpha = theta(2);
      beta = theta(3);
      q = 2*kappa*alpha/beta^2 - 1;
      c = 2*kappa/(beta^2*(1-exp(-kappa*dt)));
      u = c*x0*exp(-kappa*dt);
      v = c*x;
      p = (2*c)*ncx2pdf(2*v,2*(q+1),2*u);
   case 'bessel'
      % see Revuz and Yor (1999) p.441
      kappa = theta;  % =delta in R&Y, 1999
      nu = kappa/2 - 1;
      p = 1/(2*dt)*(x/x0).^(nu/2).*exp(-(x+x0)/(2*dt)).*...
         besseli(nu,sqrt(x*x0)/dt);
   otherwise
      error(...
         '''model'' must be one of the prescribed models (e.g. ''cir''');
end



