function [p,x,c] = densityShojiOzaki(model,x,x0,dt,theta)

% DENSITYSHOJIOZAKI  Estimate the transition density for an SDE using the
%    Shoji and Ozaki's local linearisation method described in [1].  
%
% Example:
%    f = densityShojiOzaki('CIR',linspace(.05,.15,100),0.1,1/12,[.5 .06 .15]);
%
% References:
%   [1] Shoji, I. and Ozaki, T. (1998) Estimation for nonlinear stochastic 
%       differential equations by a local linearization method, Stochastic 
%       Analysis and Applications.

% Written by Simon Preston (http://www.maths.nott.ac.uk/~spp), 2009

switch lower(model)
   case 'ou'
      [y,y0,sigmaOfx] = gammaTransform(model,x,x0,theta);
      [a0, a1, a2, a3, a4] = ou_t(y0,theta);
      p = calcDensity(y,y0,dt,a0,a1,a2)./sigmaOfx; 
   case 'cir'
      [y,y0,sigmaOfx] = gammaTransform(model,x,x0,theta);
      [a0, a1, a2, a3, a4] = cir_t(y0,theta);
      p = calcDensity(y,y0,dt,a0,a1,a2)./sigmaOfx;
   case 'bessel'
      [y,y0,sigmaOfx] = gammaTransform(model,x,x0,theta);
      [a0, a1, a2, a3, a4] = bessel_t(y0,theta);
      p = calcDensity(y,y0,dt,a0,a1,a2)./sigmaOfx;
   case 'ginzland'
      [y,y0,sigmaOfx] = gammaTransform(model,x,x0,theta);
      [a0, a1, a2, a3, a4] = ginzland_t(y0,theta);
      p = calcDensity(y,y0,dt,a0,a1,a2)./sigmaOfx;
   otherwise
      p = zeros(size(x));
end

end

% ----------------------------------------------------------------------
function p = calcDensity(x,x0,dt,a0,a1,a2,sigma)
sigma = 1;
L = a1;
M = 1/2*sigma^2*a2;
p = normpdf(x,...
   x0 + a0/L*(exp(L*dt)-1) + M/L^2*(exp(L*dt)-1-L*dt),...
   sqrt(sigma^2*(exp(2*L*dt)-1)/(2*L)));
end

% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = bessel_t(x,theta)
kappa = theta; 
[a0, a1, a2, a3, a4] = deal(...
   (kappa - 1)/2*x^(-1),...
   -(kappa - 1)/2*x^(-2),...
   (kappa - 1)*x^(-3),...
   -3*(kappa - 1)*x^(-4),...
   12*(kappa - 1)*x^(-5));
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = cir_t(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
foo = 2*kappa*alpha/beta^2-1/2;
[a0, a1, a2, a3, a4] = deal(...
   foo/x - kappa/2*x, ...
   -foo/x^2 - kappa/2, ...
   2*foo/x^3, ...
   -6*foo/x^4, ...
   24*foo/x^5);
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = ginzland_t(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4] = deal(...
   -kappa*x - alpha*beta^2*x^3, ...
   -kappa - 3*alpha*beta^2*x^2, ...
   -6*alpha*beta^2*x, ...
   -6*alpha*beta^2, ...
   0);
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = ou_u(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = deal(...
   kappa*(alpha-x), ...
   -kappa, ...
   0,0,0, ...
   beta,0,0,0,0);
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = ou_t(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4] = deal(...
   kappa*alpha/beta-kappa*x, ...
   -kappa,0,0,0);
end
