function [p,x] = densityTrunc(model,x,x0,dt,theta,method)

% DENSITYTRUNC  Estimate the transition density for an SDE using the
% density of a truncated Ito-Taylor series expansion.  Input "method" can
% be one of 'euler_u' (Euler scheme without transform to unit diffusion),
% 'euler_t' (Euler scheme with transform), or 'elerian_u' (Milstein scheme 
% without transform; see [1]).
%
% Example:
%     f = densityTrunc('CIR',linspace(.05,.15,100),0.1,1/12,[.5 .06 .15],'elerian_u');
%
% Reference: 
%    [1] Elerian, O. (1998) A Note on the Existence of a Closed Form 
%        Conditional Transition Density for the Milstein Scheme, Working 
%        Paper, Nuffield College, Oxford University.

% Written by Simon Preston (http://www.maths.nott.ac.uk/~spp), 2009

f = densitySP('CIR',x,0.1,1/12,[.5 .06 .15],'scheme4_t');


% DENSITYSP  Estimate the transition density for an SDE using the
%    saddlepoint method described in [1].
%
%    Inputs:
%    model: one of {'bessel','sin','blackscholes','cir','ginzland',
%            'ou','log', 'gcir'} [see README.txt for how to define other
%            models]
%    x: value at the right-hand side of interval [scalar or vector]
%    x0: value at the left-hand side of interval [scalar]
%    dt: interval size
%    theta: parameter vector
%    method: one of {'scheme3_u','scheme4_t'} 
%
%    Example: 
%    x = linspace(.05,.15,100);
%    f = densitySP('CIR',x,0.1,1/12,[.5 .06 .15],'scheme4_t');
%    plot(x,f);
%
% Reference:
%    [1] Preston, S.P and Wood, A.T.A (2012) Approximation of transition
%        densities of stochastic differential equations by saddlepoint 
%        methods applied to small-time Ito-Taylor sample-path expansions,
%        Stat. Comput., 22 (2012), pp. 205-217

% Written by Simon Preston (http://www.maths.nott.ac.uk/~spp), 2009

fh = str2func([ lower(model) , '_', lower(method(end))]);

% transform data if 'method' requires it, and calc coeffs
if strcmpi(method(end),'t')
   [y,y0,sigmaOfx] = gammaTransform(model,x,x0,theta);
   [a0, a1, a2, a3, a4] = fh(y0,theta); b0 = 1; b1 = 0; b2 = 0;
else
   [y,y0,sigmaOfx] = deal(x,x0,1);
   [a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = fh(y0,theta);
end


switch method(1:end-2)
   case 'euler'
      mu = y0+a0*dt;  sigma = b0*sqrt(dt);
      py = 1/(sqrt(2*pi)*sigma) * exp(-(y-mu).^2/(2*sigma^2));
      % py = normpdf(y,y0+a0*dt,b0*sqrt(dt));
   case 'elerian'
      if b1 == 0,
         warning('Elerian''s density is not defined for b = const') %#ok<*WNTAG>
         p = NaN(size(y));
         return
      end
      lambda = 1/(dt*b1^2);
      A = b0*b1*dt/2;
      B = -b0/(2*b1) + y0 + a0*dt - b0*b1*dt/2;
      z = (y - B)/A;
      if any(z<=0)
         warning('Elerian''s density is defined only on positive real line');
         p = NaN(size(y));
         return
      end
      %py1 = exp(-lambda/2)/(sqrt(2*pi)*abs(A))*z.^(-1/2).*exp(-z/2).*...
      %   cosh(sqrt(lambda*z));
      py = exp(-lambda/2)/(sqrt(2*pi)*abs(A))*z.^(-1/2).*...
         0.5.*(exp(-z/2+sqrt(lambda*z)) + exp(-z/2-sqrt(lambda*z)));
   case 'kessler2'
      mu = y0 + a0*dt + (a0*a1 + b0^2*a2/2)*dt^2/2;
      sigma_sq = y0^2 + (2*a0*y0 + b0^2)*dt + (2*a0*(a1*y0+a0+b0*b1) + ...
         b0^2*(a2*y0+2*a1+b1^2+b0*b2))*dt^2/2 - mu^2;
      if sigma_sq < 0
         warning('Variance of Kessler''s method is negative');
      end
      py = normpdf(y,mu,sqrt(sigma_sq));
    otherwise
        error('the specified method is not a valid one for densityTrunc');
end

p = py./sigmaOfx;

end

% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = bessel_t(x,theta)
kappa = theta;
[a0, a1, a2, a3, a4] = deal(...
   (kappa - 1)/2*x^(-1),...
   -(kappa - 1)/2*x^(-2),...
   (kappa - 1)*x^(-3),...
   -3*(kappa - 1)*x^(-4),...
   12*(kappa - 1)*x^(-5));
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = bessel_u(x,theta)
kappa = theta;
[a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = deal(...
   kappa,0,0,0,0,...
   2*x^(1/2),...
   x^(-1/2),...
   -1/2*x^(-3/2),...
   3/4*x^(-5/2),...
   -15/8*x^(-7/2));
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = cir_t(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
foo = 2*kappa*alpha/beta^2-1/2;
[a0, a1, a2, a3, a4] = deal(...
   foo/x - kappa/2*x, ...
   -foo/x^2 - kappa/2, ...
   2*foo/x^3, ...
   -6*foo/x^4, ...
   24*foo/x^5);
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = cir_u(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = deal(...
   kappa*(alpha-x), ...
   -kappa, ...
   0, 0, 0, ...
   beta*x.^(1/2), ...
   1/2*beta*x.^(-1/2), ...
   -1/4*beta*x.^(-3/2), ...
   3/8*beta*x.^(-5/2), ...
   -15/16*beta*x.^(-7/2));
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = ginzland_u(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = deal(...
   -kappa*x - alpha*x^3, ...
   -kappa - 3*alpha*x^2, ...
   -6*alpha*x, ...
   -6*alpha, ...
   0,...
   beta,0,0,0,0);
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = ginzland_t(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4] = deal(...
   -kappa*x - alpha*beta^2*x^3, ...
   -kappa - 3*alpha*beta^2*x^2, ...
   -6*alpha*beta^2*x, ...
   -6*alpha*beta^2, ...
   0);
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = log_t(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4] = deal(...
   beta/2 - kappa/beta + kappa/(beta*alpha)*exp(-beta*x), ...
   -kappa/alpha*exp(-beta*x), ...
   beta*kappa/alpha*exp(-beta*x), ...
   -beta^2*kappa/alpha*exp(-beta*x), ...
   beta^3*kappa/alpha*exp(-beta*x));
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = ou_t(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4] = deal(...
   kappa*alpha/beta-kappa*x, ...
   -kappa,0,0,0);
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = ou_u(x,theta)
kappa = theta(1); alpha = theta(2); beta = theta(3);
[a0, a1, a2, a3, a4, b0, b1, b2, b3, b4] = deal(...
   kappa*(alpha-x), ...
   -kappa, ...
   0,0,0, ...
   beta,0,0,0,0);
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = sin_t(x,theta)
alpha = theta;
[a0, a1, a2, a3, a4] = deal(...
   sin(x-alpha), ...
   cos(x-alpha), ...
   -sin(x-alpha), ...
   -cos(x-alpha), ...
   sin(x-alpha));
end
% ----------------------------------------------------------------------
function [a0, a1, a2, a3, a4] = toy_t(x,theta)
[a0, a1, a2, a3, a4] = deal(theta,0,0,0,0);
end
