function [X,t] = traj(model,t,theta,X0,n,scheme)

% TRAJ Generate trajectories from an SDE.
%    INPUTS - 
%    model: one of {'bessel','sin','blackscholes','cir','ginzland',
%            'ou','log', 'gcir'}
%    t: vector of times at which to evaluate the trajectory
%    theta: parameter vector
%    X0: initial value of process, i.e. at time t(1)
%    n: number of trajectories to generate  [default = 1]
%    scheme: one of {'euler','milstein','exact'},
%            [default = 'milstein', NB exact available here only for 'cir']
%    OUTPUTS - each of the n columns of X is a trajectory corresponding to 
%    time vector t.

%   Written by Simon Preston (http://www.maths.nott.ac.uk/~spp), 2009
%   [except for 'exact' simulation of CIR process for which the function 
%   'cirpath' is called, a function written by Dimitri Shvorob 
%   (dimitri.shvorob@vanderbilt.edu)]

if nargin<2||isempty(t), t = linspace(0,10,1000); end
if nargin<5||isempty(n), n = 1; end
if nargin<6||isempty(n), scheme = 'milstein' ; end
if size(t,1) < size(t,2), t = t'; end

% --- Check input ------------------------------------------------------
switch lower(model)
    case {'bessel','sin'}, nParams = 1;
    case {'blackscholes'}, nParams = 2;
    case {'cir','ginzland','ou','log'}, nParams = 3;
    case {'gcir'}, nParams = 4;
    otherwise, error(['Specified model ''' model ...
            ''' is not covered by this code'])
end
if length(theta)~=nParams, error(['Specified model ''' model ...
        ''' needs ' num2str(nParams) ' parameters, but ' ...
        num2str(length(theta)) ' have been given']);
end
% ----------------------------------------------------------------------

numDivs = length(t);
X = repmat([X0; zeros(numDivs-1,1)],1,n);
dt = repmat(diff(t),1,n);
dB = randn(numDivs-1,n) .* sqrt(dt);
dBsq = dB.^2;

fh = str2func(model);

switch scheme
    case 'milstein'
        for i = 1:(numDivs-1)
            [a0, b0, b1] = fh(X(i,:),theta);
            X(i+1,:) = X(i,:) + a0.*dt(i,:) + b0.*dB(i,:) + ...
                0.5*b0.*b1.*(dBsq(i,:)-dt(i,:));
        end
    case 'euler'
        for i = 1:(numDivs-1)
            [a0, b0] = fh(X(i,:),theta);
            if ~isreal(a0) || ~isreal(b0), error; end; %#ok<LTARG>
            X(i+1,:) = X(i,:) + a0.*dt(i,:) + b0.*dB(i,:);
        end
    case 'exact'
        switch model
            case 'cir'
                for iN = 1:n
                    X(:,iN) = cirpath(t,theta(1),theta(2),theta(3),X0);
                end
            otherwise
                error(['Exact simulation not coded for model ''' model ''''])
        end
end

end

% ----------------------------------------------------------------------
function [a0, b0, b1] = bessel(x,theta)
[a0, b0, b1] = deal(theta(1), 2*sqrt(x), x.^(-0.5));
end
% ----------------------------------------------------------------------
function [a0, b0, b1] = blackscholes(x,theta)
[a0, b0, b1] = deal(theta(1)*x, theta(2)*x, theta(2));
end
% ----------------------------------------------------------------------
function [a0, b0, b1] = cir(x,theta)
[a0, b0, b1] = deal(...
    theta(1)*(theta(2)-x),...
    theta(3)*x.^(0.5),...
    0.5*theta(3)*x.^(-0.5));
end
% ----------------------------------------------------------------------
function [a0, b0, b1] = gcir(x,theta)
[a0, b0, b1] = deal(...
    theta(1)*(theta(2)-x),...
    theta(3)*x.^theta(4),...
    theta(4)*theta(3)*x.^(theta(4)-1));
end
% ----------------------------------------------------------------------
function [a0, b0, b1] = ginzland(x,theta)
[a0, b0, b1] = deal(...
    -theta(1)*x-theta(2)*x.^3, ...
    theta(3),...
    0);
end
% ----------------------------------------------------------------------
function [a0, b0, b1] = log(x,theta)
[a0, b0, b1] = deal(...
    theta(1).*x.*(1-x/theta(2)),...
    theta(3).*x,...
    theta(3));
end
% ----------------------------------------------------------------------
function [a0, b0, b1] = ou(x,theta)
[a0, b0, b1] = deal(theta(1)*(theta(2)-x),theta(3),0);
end
% ----------------------------------------------------------------------
function [a0, b0, b1] = sin(x,theta)
[a0, b0, b1] = deal(sin(x-theta),1,0);
end